import request from 'supertest';
import { BigNumber, ethers } from 'ethers';
import { HttpStatus, INestApplication } from '@nestjs/common';
import { getContractAddresses } from '@contracts';
import { getPublicApp, getSigners } from '../utils/setup';

let app: INestApplication;
let signerWithAddress: any;

describe('Test public payment endpoints (e2e)', () => {
  beforeAll(async () => {
    app = await getPublicApp();
    await app.init();
    signerWithAddress = await getSigners();
  });

  it(`/GET /payment/approve-operator-to-spend-payment-tokens - get correct approval for spending tokens (needs to be signed and sent by approver)`, async () => {
    const response = await request(app.getHttpServer())
      .get('/payment/approve-operator-to-spend-payment-tokens')
      .expect(200);

    const data = response.body;

    const { contractAddressPaymentToken } = getContractAddresses(
      process.env.HARDHAT_NETWORK,
    );
    expect(data).toBeTruthy();
    expect(data).toHaveProperty('to');
    expect(ethers.utils.isAddress(data.to)).toBeTruthy();
    expect(data.to).toEqual(`${contractAddressPaymentToken}`);
    expect(response.statusCode).toBe(HttpStatus.OK);
  });

  it('/GET /payment/get-increase-allowance-tx - should return a valid unsigned tx to payment token contract with signer[1] as spender', async () => {
    const toApproveAddress = signerWithAddress[1].address;
    const amountToAllow = '1000000000000000000';

    const response = await request(app.getHttpServer())
      .get('/payment/get-increase-allowance-tx')
      .query({ toApproveAddress, amountToAllow });

    expect(response.status).toBe(HttpStatus.OK);

    const rawTx = response.body;

    expect(rawTx).toHaveProperty('to');
    expect(rawTx).toHaveProperty('data');
    expect(ethers.utils.isAddress(rawTx.to)).toBe(true);

    const { contractAddressPaymentToken } = getContractAddresses('localhost');

    expect(rawTx.to.toLowerCase()).toBe(
      contractAddressPaymentToken.toLowerCase(),
    );
    expect(rawTx.data.toLowerCase()).toContain(
      toApproveAddress.toLowerCase().slice(2),
    );
  });

  it('/GET /payment/get-payment-allowance-on-address - should return updated allowance after sending generated tx', async () => {
    const owner = signerWithAddress[1];
    const spender = signerWithAddress[2];
    const amountToAllow = '2000000000000000000'; // 2 tokens

    // Step 1: Get unsigned tx from API
    const unsignedTxResponse = await request(app.getHttpServer())
      .get('/payment/get-increase-allowance-tx')
      .query({
        toApproveAddress: spender.address,
        amountToAllow,
      })
      .expect(HttpStatus.OK);

    const unsignedTx = unsignedTxResponse.body;

    // Step 2: Send the tx using the owner signer
    const txResponse = await owner.sendTransaction({
      to: unsignedTx.to,
      data: unsignedTx.data,
      gasLimit: unsignedTx.gasLimit ?? 100000,
      gasPrice: unsignedTx.gasPrice ?? undefined,
      value: unsignedTx.value ?? 0,
    });

    await txResponse.wait();

    // Step 3: Check allowance via endpoint
    const allowanceResponse = await request(app.getHttpServer())
      .get('/payment/get-payment-allowance-on-address')
      .query({
        userAddress: owner.address,
        allowanceCheckAddress: spender.address,
      })
      .expect(HttpStatus.OK);

    const allowance = allowanceResponse.text;

    expect(typeof allowance).toBe('string');
    expect(BigNumber.from(allowance).gt(BigNumber.from('1'))).toBe(true);
  });

  afterAll(async () => {
    await app.close();
  });
});
